-- ============================================
-- BonusHelper2026 - Pending Operations Table
-- ============================================
-- Admin onay sistemi için kullanıcı işlemlerinin bekletildiği tablo
-- JSON tabanlı operation data storage
-- Çalıştırma: 01_Create_AYARLAR_Table.sql'den sonra
-- Sıra: 2. Script

USE [BONUS_HELPER]
GO

-- Eski PENDING_OPERATIONS tablosunu sil (varsa - eski şemayı güncelliyoruz)
IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[PENDING_OPERATIONS]') AND type in (N'U'))
BEGIN
    PRINT 'Eski PENDING_OPERATIONS tablosu siliniyor...'
    DROP TABLE [dbo].[PENDING_OPERATIONS]
    PRINT '✓ Eski tablo silindi.'
END
GO

-- Yeni PENDING_OPERATIONS tablosu (JSON tabanlı)
CREATE TABLE [dbo].[PENDING_OPERATIONS](
    [ID] [int] IDENTITY(1,1) NOT NULL,
    
    -- Talep eden kullanıcı bilgileri
    [FK_USER_ID] [int] NOT NULL,
    [USER_NAME] [nvarchar](50) NOT NULL,
    
    -- İşlem tipi ve JSON data
    [OPERATION_TYPE] [varchar](50) NOT NULL,
    [OPERATION_DATA] [nvarchar](max) NOT NULL,
    
    -- Durum ve tarihler
    [STATUS] [tinyint] NOT NULL DEFAULT 0,
    [REQUEST_DATE] [datetime] NOT NULL DEFAULT GETDATE(),
    
    -- Onay/Red bilgileri
    [APPROVED_BY] [nvarchar](50) NULL,
    [APPROVED_DATE] [datetime] NULL,
    [REJECT_REASON] [nvarchar](500) NULL,
    
    -- Ek bilgiler
    [NOTES] [nvarchar](500) NULL,
    [ERROR_MESSAGE] [nvarchar](max) NULL,
    
 CONSTRAINT [PK_PENDING_OPERATIONS] PRIMARY KEY CLUSTERED ([ID] ASC)
)
GO

-- Foreign Key (KULLANICILAR tablosuna)
IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[KULLANICILAR]') AND type in (N'U'))
BEGIN
    ALTER TABLE [dbo].[PENDING_OPERATIONS] WITH CHECK 
    ADD CONSTRAINT [FK_PENDING_OPERATIONS_KULLANICILAR] FOREIGN KEY([FK_USER_ID])
    REFERENCES [dbo].[KULLANICILAR] ([KullaniciID])
    ON UPDATE CASCADE
    ON DELETE CASCADE
    
    PRINT '✓ Foreign key KULLANICILAR tablosuna eklendi.'
END
ELSE
BEGIN
    PRINT '⚠ KULLANICILAR tablosu bulunamadı - Foreign key eklenemedi.'
    PRINT '  Önce 01_Create_AYARLAR_Table.sql çalıştırın.'
END
GO

-- Performance indeksleri
CREATE NONCLUSTERED INDEX [IX_PENDING_OPERATIONS_STATUS] 
ON [dbo].[PENDING_OPERATIONS]([STATUS] ASC)
GO

CREATE NONCLUSTERED INDEX [IX_PENDING_OPERATIONS_FK_USER_ID] 
ON [dbo].[PENDING_OPERATIONS]([FK_USER_ID] ASC)
GO

CREATE NONCLUSTERED INDEX [IX_PENDING_OPERATIONS_REQUEST_DATE] 
ON [dbo].[PENDING_OPERATIONS]([REQUEST_DATE] DESC)
GO

-- Check Constraints
ALTER TABLE [dbo].[PENDING_OPERATIONS] WITH CHECK 
ADD CONSTRAINT [CK_PENDING_OPERATIONS_STATUS] 
CHECK ([STATUS] IN (0, 1, 2, 3))
GO

ALTER TABLE [dbo].[PENDING_OPERATIONS] WITH CHECK 
ADD CONSTRAINT [CK_PENDING_OPERATIONS_OPERATION_TYPE] 
CHECK ([OPERATION_TYPE] IN (
    'PUAN_EKLE',
    'PUAN_CIKAR', 
    'PUAN_ESITLE',
    'KART_AC',
    'EXCEL_IMPORT'
))
GO

PRINT ''
PRINT '============================================'
PRINT '✓ PENDING_OPERATIONS tablosu başarıyla oluşturuldu.'
PRINT '============================================'
PRINT ''
PRINT 'Tablo Şeması:'
PRINT '  - ID: Primary Key'
PRINT '  - FK_USER_ID: İşlemi talep eden kullanıcı (FK → KULLANICILAR)'
PRINT '  - USER_NAME: Kullanıcı adı (denormalized)'
PRINT '  - OPERATION_TYPE: İşlem tipi'
PRINT '      * PUAN_EKLE - Puan ekleme'
PRINT '      * PUAN_CIKAR - Puan çıkarma'
PRINT '      * PUAN_ESITLE - Puan eşitleme'
PRINT '      * KART_AC - Kart oluşturma'
PRINT '      * EXCEL_IMPORT - Excel toplu import'
PRINT '  - OPERATION_DATA: JSON formatında işlem detayları'
PRINT '  - STATUS: İşlem durumu'
PRINT '      * 0 = Bekliyor (Pending)'
PRINT '      * 1 = Onaylandı (Approved)'
PRINT '      * 2 = Reddedildi (Rejected)'
PRINT '      * 3 = Hata (Error)'
PRINT '  - REQUEST_DATE: Talep tarihi'
PRINT '  - APPROVED_BY: Onaylayan/Reddeden admin kullanıcı adı'
PRINT '  - APPROVED_DATE: Onay/Red tarihi'
PRINT '  - REJECT_REASON: Red sebebi açıklaması'
PRINT '  - NOTES: İsteğe bağlı notlar'
PRINT '  - ERROR_MESSAGE: Hata mesajı (STATUS=3 için)'
PRINT ''
PRINT 'Örnek OPERATION_DATA (JSON):'
PRINT '  PUAN_EKLE: {"BaslangicKartNo":"1000","BitisKartNo":"1010","Puan":100,"Aciklama":"..."}'
PRINT '  KART_AC: {"BaslangicKartNo":"2000","BitisKartNo":"2099","MusteriAdi":"...","Telefon":"...","BaslangicPuani":50,...}'
PRINT ''
PRINT 'Sonraki adım: BonusHelper.exe uygulamasını başlatın'
PRINT ''
GO
